// Command line tool to tweak JPEG and TIFF files.
//
// Command syntax:
//
//   -- any text			(comment, extends to end of line)
//
//   read "file path"
//   write "file path"		(correct IIM digest if IIM is present, else no IIM digest)
//   write-nd "file path"	(no IIM digest)
//   write-zd "file path"	(zero IIM digest)
//   write-fd "file path"	(fake IIM digest)
//
//   strip [ exif | iim | xmp | psir ]
//   delete [ exif | iim | psir ] tag-number
//   delete xmp [ prefix:* | prefix:property ]
//
//   set [ exif | iim | xmp | all ] item-name item-value
//
//   set usercomment [ ASCII | BE-BOM | BE+BOM | LE-BOM | LE+BOM ] "value"
//
// A read command with a file extension of ".*" processes both ".jpg" and ".tif", applying all
// subsequent commands to the next write (or write-nd, etc.) to each. In this case the extension
// should be omitted from the path in the write command.
//
// The item-name and item-value are an MWG name and appropriate value:
//
//   description "value"
//   copyright   "value"
//   creator     "value" or (  "value"  "value" ... )
//   keywords    (  "value"  "value" ... )
//   c-date      [ RIGHT | WRONG | value in ISO 8601 form ]
//   d-date      [ RIGHT | WRONG | value in ISO 8601 form ]
//   m-date      [ RIGHT | WRONG | value in ISO 8601 form ]
//   dates       [ RIGHT | WRONG | value in ISO 8601 form ]
//   rating      value
//   orientation value
//
// Quotes for values are optional, if missing values are whitespace terminated. Quotes can be used
// inside values by doubling them, e.g. "He said ""Do it!""".

// =================================================================================================

#include <vector>
#include <string>

#include <stdio.h>
#include <stdlib.h>

#define TXMP_STRING_TYPE	std::string

#include "XMP.hpp"
#include "XMP.incl_cpp"

#include "TIFF_Support.hpp"
#include "PSIR_Support.hpp"
#include "IPTC_Support.hpp"
#include "Reconcile_Impl.hpp"
#include "MD5.h"

using namespace std;

#if XMP_WinBuild
	#pragma warning ( disable : 4244 )	// integer shortening, possible loss of data
	#pragma warning ( disable : 4267 )	// integer shortening, possible loss of data
	#pragma warning ( disable : 4996 )	// '...' was declared deprecated
#endif

// =================================================================================================

enum { kNoIIMDigest = 0, kZeroIIMDigest = 1, kNormalIIMDigest = 2, kFakeIIMDigest = 3 };

enum { kFormExif = 1, kFormIIM = 2, kFormXMP = 4 };
static const char * kFormNames[] = { "", "Exif", "IIM", "", "XMP", "", "", "all" };

enum { kFileNone = 0, kFileJPEG = 1, kFileTIFF = 2 };
static XMP_Uns8 fileType = kFileNone;
static XMP_Uns32 fileLen = 0;
static XMP_Uns8 * fileContent = 0;

static XMP_Uns32 jpegTailOffset;

static TIFF_FileWriter *	tiffManager = 0;
static PSIR_FileWriter *	psirManager = 0;
static IPTC_Writer *		iimManager  = 0;
static SXMPMeta *			xmpManager  = 0;

static const char * kExifSignatureString = "Exif\x00\x00";
static const size_t kExifSignatureLength = 6;

static const char * kPSIRSignatureString = "Photoshop 3.0\x00";
static const size_t kPSIRSignatureLength = 14;

static const char * kMainXMPSignatureString = "http://ns.adobe.com/xap/1.0/\x00";
static const size_t kMainXMPSignatureLength = 29;

static const char * kExtXMPSignatureString = "http://ns.adobe.com/xmp/extension/\x00";
static const size_t kExtXMPSignatureLength = 35;

// =================================================================================================
// =================================================================================================
// Basic utility routines   (Yes, horribly inefficient use of strings, but who cares?)
// =================================================================================================
// =================================================================================================

static void StripWhitespace ( std::string & cmdStream )
{

	while ( ! cmdStream.empty() ) {
		if ( (cmdStream[0] != ' ')  && (cmdStream[0] != '\t') &&
			 (cmdStream[0] != '\n') && (cmdStream[0] != '\r') ) break;
		cmdStream.erase ( 0, 1 );
	}

}	// StripWhitespace

// =================================================================================================

static std::string GetWord ( std::string & cmdStream )
{
	std::string word;
	
	StripWhitespace ( cmdStream );
	if ( cmdStream.empty() ) return word;
	
	size_t len = 0;
	while ( len < cmdStream.size() ) {
		if ( (cmdStream[len] == ' ')  || (cmdStream[len] == '\t') ||
			 (cmdStream[len] == '\n') || (cmdStream[len] == '\r') ) break;
		++len;
	}
	
	word.assign ( cmdStream.c_str(), len );
	cmdStream.erase ( 0, len );
	return word;

}	// GetWord

// =================================================================================================

static XMP_Uns32 GetUInt ( std::string & cmdStream )
{
	
	std::string text = GetWord ( cmdStream );
	if ( text.empty() ) return 0;
	XMP_Uns32 bin = (XMP_Uns32) strtoul ( text.c_str(), 0, 0 );
	
	return bin;
	
}	// GetUInt

// =================================================================================================

static float GetFloat ( std::string & cmdStream )
{
	
	std::string text = GetWord ( cmdStream );
	if ( text.empty() ) return 0.0;
	float bin = (float) strtod ( text.c_str(), 0 );
	
	return bin;
	
}	// GetUInt

// =================================================================================================

static std::string GetString ( std::string & cmdStream )
{
	std::string str;
	
	StripWhitespace ( cmdStream );
	if ( cmdStream.empty() ) return str;
	
	if ( cmdStream[0] != '"' ) {

		str = GetWord ( cmdStream );

	} else {

		cmdStream.erase ( 0, 1 );

		while ( ! cmdStream.empty() ) {
			if ( cmdStream[0] != '"' ) {
				str += cmdStream[0];
				cmdStream.erase ( 0, 1 );
			} else if ( (cmdStream.size() >= 2) && (cmdStream[1] == '"') ) {
				str += '"';
				cmdStream.erase ( 0, 2 );
			} else {
				cmdStream.erase ( 0, 1 );
				break;
			}
		}

	}
	
	return str;
	
}	// GetString

// =================================================================================================

static std::vector<std::string> GetStringList ( std::string & cmdStream )
{
	std::vector<std::string> list;
	std::string item;
	
	StripWhitespace ( cmdStream );
	if ( cmdStream.empty() ) return list;
	
	if ( cmdStream[0] != '(' ) {

		item = GetString ( cmdStream );
		list.push_back ( item );

	} else {

		cmdStream.erase ( 0, 1 );

		while ( ! cmdStream.empty() ) {

			StripWhitespace ( cmdStream );
			if ( cmdStream.empty() ) break;

			if ( cmdStream[0] == ')' ) { cmdStream.erase ( 0, 1 ); break; }

			item = GetString ( cmdStream );
			list.push_back ( item );

		}

	}
	
	return list;
	
}	// GetStringList

// =================================================================================================
// =================================================================================================
// Command routines
// =================================================================================================
// =================================================================================================

static void ParseJPEG()
{

	// Parse the JPEG file and save 4 parts:
	//   1. The Exif APP1 section, parsed into a TIFF manager.
	//   2. The XMP APP1 section, parsed into an SXMPMeta object.
	//   3. The Photoshop APP13 section, parsed into a PSIR manager.
	//   4. The file tail, everything after the last APPn section.
	
	XMP_Uns8 * jpegPtr   = fileContent + 2;	// Skip the initial SOI marker.
	XMP_Uns8 * jpegLimit = fileContent + fileLen;
	
	bool haveExtXMP = false;
	
	while ( jpegPtr < jpegLimit ) {
	
		XMP_Uns16 segID  = GetUns16BE ( jpegPtr );
		XMP_Uns16 segLen = GetUns16BE ( jpegPtr+2 );	// Excludes ID, includes length and data.
		XMP_Uns8 * dataPtr = jpegPtr + 4;
		
		if ( segID == 0xFFE1 ) {	// APP1
		
			if ( 0 == memcmp ( dataPtr, kExifSignatureString, kExifSignatureLength ) ) {

				tiffManager = new TIFF_FileWriter;
				tiffManager->ParseMemoryStream ( (dataPtr + kExifSignatureLength),
												 (segLen - 2 - kExifSignatureLength),
												 false /* don't copy data */ );

			} else if ( 0 == memcmp ( dataPtr, kMainXMPSignatureString, kMainXMPSignatureLength ) ) {

				xmpManager = new SXMPMeta;
				xmpManager->ParseFromBuffer ( (char*)(dataPtr + kMainXMPSignatureLength),
											  (segLen - 2 - kMainXMPSignatureLength) );

			} else if ( 0 == memcmp ( dataPtr, kExtXMPSignatureString, kExtXMPSignatureLength ) ) {

				if ( ! haveExtXMP ) printf ( "  ** Extended XMP is not currently preserved.\n" );
				haveExtXMP = true;

			}
		
		} else if ( segID == 0xFFED ) {	// APP13
		
			if ( 0 == memcmp ( dataPtr, kPSIRSignatureString, kPSIRSignatureLength ) ) {

				psirManager = new PSIR_FileWriter;
				psirManager->ParseMemoryResources ( (dataPtr + kPSIRSignatureLength),
													(segLen - 2 - kPSIRSignatureLength),
													false /* don't copy data */ );

				PSIR_Manager::ImgRsrcInfo iimInfo;
				bool haveIIM = psirManager->GetImgRsrc ( kPSIR_IPTC, &iimInfo );
				if ( haveIIM ) {
					iimManager = new IPTC_Writer;
					iimManager->ParseMemoryDataSets ( iimInfo.dataPtr, iimInfo.dataLen, false /* don't copy data */ );
				}

			}

		} else if ( (segID >> 4) != 0xFFE ) {

			break;	// Ignore other APPn, exit loop on anything else.

		}
		
		jpegPtr += (2 + (XMP_Uns32)segLen);	// Avoid 16-bit add for "2 + segLen".
		
	}
	
	jpegTailOffset = jpegPtr - fileContent;

}	// ParseJPEG

// =================================================================================================

static void ParseTIFF()
{

	// Let the TIFF manager parse and maintain everything.

	tiffManager = new TIFF_FileWriter;
	if ( tiffManager == 0 ) { printf ( "  ** Can't create tiffManager.\n" ); return; }
	
	tiffManager->ParseMemoryStream ( fileContent, fileLen, false /* don't copy data */ );
	
	TIFF_Manager::TagInfo tagInfo;
	bool haveTag;
	
	haveTag = tiffManager->GetTag ( kTIFF_PrimaryIFD, kTIFF_IPTC, &tagInfo );
	if ( haveTag ) {
		iimManager = new IPTC_Writer;
		iimManager->ParseMemoryDataSets ( tagInfo.dataPtr, tagInfo.dataLen, false /* don't copy data */ );
	}
	
	haveTag = tiffManager->GetTag ( kTIFF_PrimaryIFD, kTIFF_PSIR, &tagInfo );
	if ( haveTag ) {
		psirManager = new PSIR_FileWriter;
		psirManager->ParseMemoryResources ( tagInfo.dataPtr, tagInfo.dataLen, false /* don't copy data */ );
	}
	
	haveTag = tiffManager->GetTag ( kTIFF_PrimaryIFD, kTIFF_XMP, &tagInfo );
	if ( haveTag ) {
		xmpManager = new SXMPMeta;
		xmpManager->ParseFromBuffer ( (char*)tagInfo.dataPtr, tagInfo.dataLen );
	}
	
}	// ParseTIFF

// =================================================================================================

static bool ReadFile ( std::string & in_filePath, const char * whichFile )
{
	std::string filePath = in_filePath;

	bool multiFile = (filePath[filePath.size()-1] == '*');
	if ( multiFile ) {
		filePath.erase ( filePath.size()-1 );
		filePath += whichFile;
	}
	
	printf ( "Read file \"%s\"\n", filePath.c_str() );

	fileType = kFileNone;
	fileLen = 0;
	if ( fileContent != 0 ) { free ( fileContent ); fileContent = 0; }

	if ( tiffManager != 0 ) { delete tiffManager; tiffManager = 0; }
	if ( psirManager != 0 ) { delete psirManager; psirManager = 0; }
	if ( iimManager != 0 ) { delete iimManager; iimManager = 0; }
	if ( xmpManager != 0 ) { delete xmpManager; xmpManager = 0; }
	
	FILE * inFile = fopen ( filePath.c_str(), "rb" );
	if ( inFile == 0 ) { printf ( "  ** Can't open file.\n" ); return multiFile; }
	
	(void) fseek ( inFile, 0, SEEK_END );
	fileLen = (XMP_Uns32) ftell ( inFile );
	if ( fileLen < 4 ) { printf ( "  ** File too short.\n" ); fclose ( inFile ); return multiFile; }
	
	XMP_Uns32 first4;
	(void) fseek ( inFile, 0, SEEK_SET );
	(void) fread ( &first4, 1, 4, inFile );
	first4 = MakeUns32BE ( first4 );
	
	if ( (first4 >> 8) == 0xFFD8FF ) fileType = kFileJPEG;
	if ( (first4 == TIFF_Manager::kBigEndianPrefix) || (first4 == TIFF_Manager::kLittleEndianPrefix) ) fileType = kFileTIFF;
	
	fileContent = (XMP_Uns8*) malloc ( fileLen );
	if ( fileContent == 0 ) { printf ( "  ** Can't allocate file buffer.\n" ); fclose ( inFile ); return multiFile; }
	(void) fseek ( inFile, 0, SEEK_SET );
	(void) fread ( fileContent, 1, fileLen, inFile );
	fclose ( inFile );

	try {

		if ( fileType == kFileJPEG ) {
			ParseJPEG();
		} else if ( fileType == kFileTIFF ) {
			ParseTIFF();
		} else {
			printf ( "  ** Unrecognized file format.\n" );
			fileType = kFileNone; fileLen = 0;
			free ( fileContent ); fileContent = 0;
		}

	} catch ( ... ) {

		printf ( "  ** Parse failure\n" );
		fileType = kFileNone; fileLen = 0;
		free ( fileContent ); fileContent = 0;
		delete tiffManager; tiffManager = 0;
		delete psirManager; psirManager = 0;
		delete iimManager; iimManager = 0;
		delete xmpManager; xmpManager = 0;

	}
	
	return multiFile;
	
}	// ReadFile

// =================================================================================================

static void WriteJPEG ( FILE * outFile, int iimDigestForm )
{
	XMP_Uns32 segLen;
	XMP_Uns16 len16;
	
	fwrite ( "\xFF\xD8", 1, 2, outFile );	// Write the SOI marker.
	
	if ( tiffManager != 0 ) {	// Write the Exif APP1 section.

		void * exifPtr;
		XMP_Uns32 exifLen;

		bool compactTIFF = true;
		if ( tiffManager->GetTag ( kTIFF_ExifIFD, kTIFF_MakerNote, 0 ) ) compactTIFF = false;

		try {
			if ( ! compactTIFF ) printf ( "  -- Updating TIFF by append.\n" );
			exifLen = tiffManager->UpdateMemoryStream ( &exifPtr, compactTIFF );
		} catch ( ... ) {
			if ( compactTIFF ) {
				printf ( "  -- Updating TIFF by append.\n" );
				exifLen = tiffManager->UpdateMemoryStream ( &exifPtr, false );
			}
		}

		segLen = 2 + kExifSignatureLength + exifLen;
		if ( segLen > 65535 ) { printf ( "  ** Exif too large, %d.\n", segLen ); return; }

		fwrite ( "\xFF\xE1", 1, 2, outFile );
		len16 = MakeUns16BE ( (XMP_Uns16)segLen );
		fwrite ( &len16, 1, 2, outFile );
		fwrite ( kExifSignatureString, 1, kExifSignatureLength, outFile );
		fwrite ( exifPtr, 1, exifLen, outFile );

	}
	
	if ( xmpManager != 0 ) {	// Write the XMP APP1 section.
	
		std::string mainXMP, extXMP, extDigest;
		SXMPUtils::PackageForJPEG ( *xmpManager, &mainXMP, &extXMP, &extDigest );
		if ( ! extXMP.empty() ) { printf ( "  ** Extended XMP not supported yet.\n" ); return; }
		segLen = 2 + kMainXMPSignatureLength + mainXMP.size();
		if ( segLen > 65535 ) { printf ( "  ** XMP too large, %d.\n", segLen ); return; }

		fwrite ( "\xFF\xE1", 1, 2, outFile );
		len16 = MakeUns16BE ( (XMP_Uns16)segLen );
		fwrite ( &len16, 1, 2, outFile );
		fwrite ( kMainXMPSignatureString, 1, kMainXMPSignatureLength, outFile );
		fwrite ( mainXMP.c_str(), 1, mainXMP.size(), outFile );
	
	}

	if ( (iimManager == 0) && (psirManager != 0) ) {
		// Make sure the PSIR has no extraneous IIM or digest.
		psirManager->DeleteImgRsrc ( kPSIR_IPTC );
		psirManager->DeleteImgRsrc ( kPSIR_IPTCDigest );
	}
	
	if ( (iimDigestForm == kZeroIIMDigest) || (iimDigestForm == kFakeIIMDigest) ) {

		// Write a digest whether there is actual IIM or not.

		MD5_Digest iimDigest;
		if ( iimDigestForm == kZeroIIMDigest ) {
			memset ( &iimDigest, 0, sizeof(iimDigest) );
		} else {	// iimDigestForm == kFakeIIMDigest
			memcpy ( &iimDigest, "Fake IIM digest!", sizeof(iimDigest) );
		}

		if ( psirManager == 0 ) psirManager = new PSIR_FileWriter;
		psirManager->SetImgRsrc ( kPSIR_IPTCDigest, iimDigest, sizeof(iimDigest) );

	}
		
	if ( iimManager != 0 ) {	// Put the IIM into the PSIR.

		void * iimPtr;
		XMP_Uns32 iimLen = iimManager->UpdateMemoryDataSets ( &iimPtr );

		if ( psirManager == 0 ) psirManager = new PSIR_FileWriter;
		psirManager->SetImgRsrc ( kPSIR_IPTC, iimPtr, iimLen );

		if ( iimDigestForm == kNoIIMDigest ) {
			psirManager->DeleteImgRsrc ( kPSIR_IPTCDigest );
		} else if ( iimDigestForm == kNormalIIMDigest ) {
			MD5_Digest iimDigest;
			struct MD5_CTX md5Context;
			MD5Init ( &md5Context );
			MD5Update ( &md5Context, (XMP_Uns8*)iimPtr, iimLen );
			MD5Final ( iimDigest, &md5Context );
			psirManager->SetImgRsrc ( kPSIR_IPTCDigest, iimDigest, sizeof(iimDigest) );
		}

	}
	
	if ( psirManager != 0 ) {	// Write the PSIR APP13 section.

		void * psirPtr;
		XMP_Uns32 psirLen = psirManager->UpdateMemoryResources ( &psirPtr );
		segLen = 2 + kPSIRSignatureLength + psirLen;
		if ( segLen > 65535 ) { printf ( "  ** PSIR too large, %d.\n", segLen ); return; }

		fwrite ( "\xFF\xED", 1, 2, outFile );
		len16 = MakeUns16BE ( (XMP_Uns16)segLen );
		fwrite ( &len16, 1, 2, outFile );
		fwrite ( kPSIRSignatureString, 1, kPSIRSignatureLength, outFile );
		fwrite ( psirPtr, 1, psirLen, outFile );

	}
	
	fwrite ( &fileContent[jpegTailOffset], 1, (fileLen - jpegTailOffset), outFile );	// Write the file tail.

}	// WriteJPEG

// =================================================================================================

static void WriteTIFF ( FILE * outFile, int iimDigestForm )
{
	
	if ( xmpManager != 0 ) {	// Set the XMP tag.
		std::string xmpPkt;
		xmpManager->SerializeToBuffer ( &xmpPkt, kXMP_UseCompactFormat );
		tiffManager->SetTag ( kTIFF_PrimaryIFD, kTIFF_XMP, kTIFF_UndefinedType, xmpPkt.size(), xmpPkt.c_str() );
	} else {
		tiffManager->DeleteTag ( kTIFF_PrimaryIFD, kTIFF_XMP );
	}

	if ( iimManager == 0 ) {
		// Make sure there is no extraneous IIM or digest.
		tiffManager->DeleteTag ( kTIFF_PrimaryIFD, kTIFF_IPTC );
		if ( psirManager != 0 ) psirManager->DeleteImgRsrc ( kPSIR_IPTCDigest );
	}
	
	if ( (iimDigestForm == kZeroIIMDigest) || (iimDigestForm == kFakeIIMDigest) ) {

		// Write a digest whether there is actual IIM or not.

		MD5_Digest iimDigest;
		if ( iimDigestForm == kZeroIIMDigest ) {
			memset ( &iimDigest, 0, sizeof(iimDigest) );
		} else {	// iimDigestForm == kFakeIIMDigest
			memcpy ( &iimDigest, "Fake IIM digest!", sizeof(iimDigest) );
		}

		if ( psirManager == 0 ) psirManager = new PSIR_FileWriter;
		psirManager->SetImgRsrc ( kPSIR_IPTCDigest, iimDigest, sizeof(iimDigest) );

	}
	
	if ( iimManager != 0 ) {	// Set the IIM tag, and put the digest into the PSIR.

		void * iimPtr;
		XMP_Uns32 iimLen = iimManager->UpdateMemoryDataSets ( &iimPtr );
		tiffManager->SetTag ( kTIFF_PrimaryIFD, kTIFF_IPTC, kTIFF_UndefinedType, iimLen, iimPtr );

		if ( iimDigestForm == kNoIIMDigest ) {
			if ( psirManager != 0 ) psirManager->DeleteImgRsrc ( kPSIR_IPTCDigest );
		} else if ( iimDigestForm == kNormalIIMDigest ) {
			MD5_Digest iimDigest;
			struct MD5_CTX md5Context;
			MD5Init ( &md5Context );
			MD5Update ( &md5Context, (XMP_Uns8*)iimPtr, iimLen );
			MD5Final ( iimDigest, &md5Context );
			if ( psirManager == 0 ) psirManager = new PSIR_FileWriter;
			psirManager->SetImgRsrc ( kPSIR_IPTCDigest, iimDigest, sizeof(iimDigest) );
		}

	}
	
	if ( psirManager != 0 ) {	// Set the PSIR tag.
		void * psirPtr;
		XMP_Uns32 psirLen = psirManager->UpdateMemoryResources ( &psirPtr );
		tiffManager->SetTag ( kTIFF_PrimaryIFD, kTIFF_PSIR, kTIFF_UndefinedType, psirLen, psirPtr );
	} else {
		tiffManager->DeleteTag ( kTIFF_PrimaryIFD, kTIFF_PSIR );
	}

	void * tiffPtr;
	XMP_Uns32 tiffLen;

	bool compactTIFF = true;
	if ( tiffManager->GetTag ( kTIFF_ExifIFD, kTIFF_MakerNote, 0 ) ) compactTIFF = false;
	
	try {
		if ( ! compactTIFF ) printf ( "  -- Updating TIFF by append.\n" );
		tiffLen = tiffManager->UpdateMemoryStream ( &tiffPtr, compactTIFF );
	} catch ( ... ) {
		if ( compactTIFF ) {
			printf ( "  -- Updating TIFF by append.\n" );
			tiffLen = tiffManager->UpdateMemoryStream ( &tiffPtr, false );
		}
	}
	
	fwrite ( tiffPtr, 1, tiffLen, outFile );

}	// WriteTIFF

// =================================================================================================

static void WriteFile ( std::string & cmdStream, int iimDigestForm, const char * whichFile )
{
	std::string filePath = GetString ( cmdStream );
	if ( whichFile[0] != 0 ) {	// Non-empty when doing JPEG and TIFF wildcard.
		filePath += '.';
		filePath += whichFile;
	}
	
	printf ( "Write file \"%s\"\n", filePath.c_str() );
	
	if ( fileContent == 0 ) { printf ( "  ** No content to write.\n" ); return; }
	
	FILE * outFile = fopen ( filePath.c_str(), "wb" );
	if ( outFile == 0 ) { printf ( "  ** Can't open file.\n" ); return; }
	
	try {
		if ( fileType == kFileJPEG ) {
			WriteJPEG ( outFile, iimDigestForm );
		} else if ( fileType == kFileTIFF ) {
			WriteTIFF ( outFile, iimDigestForm );
		} else {
			printf ( "  ** Unrecognized file format.\n" );
		}
	} catch ( ... ) {
		printf ( "  ** Write failure\n" );
	}
	
	fclose ( outFile );
	
}	// WriteFile

// =================================================================================================

static void StripBlock ( std::string & cmdStream )
{
	std::string dataForm = GetWord ( cmdStream );
	printf ( "Strip \"%s\" block\n", dataForm.c_str() );
	
	if ( dataForm == "exif" ) {
	
		// Strip only the MWG-recognized tags, not all of the Exif. Keep Orientation and DateTime.
		if ( tiffManager != 0 ) {
			tiffManager->DeleteTag ( kTIFF_PrimaryIFD, kTIFF_ImageDescription );
			tiffManager->DeleteTag ( kTIFF_PrimaryIFD, kTIFF_Copyright );
			tiffManager->DeleteTag ( kTIFF_PrimaryIFD, kTIFF_Artist );
			tiffManager->DeleteTag ( kTIFF_ExifIFD, kTIFF_UserComment );
		}
	
	} else if ( dataForm == "iim" ) {
	
		// Strip the IIM. The digest will get removed by the "write" command.
		if ( iimManager != 0 ) delete iimManager;
		iimManager = 0;
	
	} else if ( dataForm == "xmp" ) {

		if ( xmpManager != 0 ) delete xmpManager;
		xmpManager = 0;
	
	} else if ( dataForm == "psir" ) {

		if ( psirManager != 0 ) delete psirManager;
		psirManager = 0;

	} else {

		printf ( "** Unrecognized form to strip \"%s\"\n", dataForm.c_str() );

	}
	
}	// StripBlock

// =================================================================================================

static void DeleteItem ( std::string & cmdStream )
{
	std::string dataForm = GetWord ( cmdStream );
	
	if ( dataForm == "exif" ) {
	
		XMP_Uns32 tagNum = GetUInt ( cmdStream );
		printf ( "Delete Exif tag %d\n", tagNum );
		if ( tiffManager != 0 ) {	// Delete the tag from all main IFDs.
			tiffManager->DeleteTag ( kTIFF_PrimaryIFD, tagNum );
			tiffManager->DeleteTag ( kTIFF_ExifIFD, tagNum );
			tiffManager->DeleteTag ( kTIFF_GPSInfoIFD, tagNum );
			tiffManager->DeleteTag ( kTIFF_TNailIFD, tagNum );
		}
	
	} else if ( dataForm == "iim" ) {
	
		XMP_Uns32 dsNum = GetUInt ( cmdStream );
		printf ( "Delete IIM DataSet %d\n", dsNum );
		if ( iimManager != 0 ) iimManager->DeleteDataSet ( dsNum );
	
	} else if ( dataForm == "psir" ) {
	
		XMP_Uns32 rsrcNum = GetUInt ( cmdStream );
		printf ( "Delete PSIR %d\n", rsrcNum );
		if ( psirManager != 0 ) psirManager->DeleteImgRsrc ( rsrcNum );
	
	} else if ( dataForm == "xmp" ) {

		std::string propName = GetWord ( cmdStream );
		printf ( "Delete XMP \"%s\"\n", propName.c_str() );
		
		if ( xmpManager != 0 ) {

			size_t colonPos = propName.find ( ':' );
			if ( colonPos == std::string::npos ) { printf ( "  ** Need qualified name.\n" ); return; }

			std::string nsURI, nsPrefix;
			nsPrefix.assign ( propName.c_str(), colonPos );
			bool found = SXMPMeta::GetNamespaceURI ( nsPrefix.c_str(), &nsURI );
			if ( ! found ) { printf ( "  ** Prefix is unknown.\n" ); return; }

			if ( propName[colonPos+1] == '*' ) {
				SXMPUtils::RemoveProperties ( xmpManager, nsURI.c_str(), 0, kXMPUtil_DoAllProperties );
			} else {
				xmpManager->DeleteProperty ( nsURI.c_str(), propName.c_str() );
			}

		}

	} else {

		printf ( "** Unrecognized form to delete \"%s\"\n", dataForm.c_str() );

	}
	
}	// DeleteItem

// =================================================================================================

static void SetDescription ( std::string & cmdStream, XMP_Uns8 formMask )
{
	std::string value = GetString ( cmdStream );
	printf ( "Set description in %s to \"%s\"\n", kFormNames[formMask], value.c_str() );
	
	if ( formMask & kFormExif ) {
		if ( tiffManager == 0 ) tiffManager = new TIFF_FileWriter;
		tiffManager->SetTag_ASCII ( kTIFF_PrimaryIFD, kTIFF_ImageDescription, value.c_str() );
	}
	
	if ( formMask & kFormIIM ) {
		if ( iimManager == 0 ) iimManager = new IPTC_Writer;
		iimManager->SetDataSet_UTF8 ( kIPTC_Description, value.c_str(), value.size() );
	}
	
	if ( formMask & kFormXMP ) {
		if ( xmpManager == 0 ) xmpManager = new SXMPMeta;
		xmpManager->SetLocalizedText ( kXMP_NS_DC, "description", "", "x-default", value.c_str() );
	}

}	// SetDescription

// =================================================================================================

static void SetCopyright ( std::string & cmdStream, XMP_Uns8 formMask )
{
	std::string value = GetString ( cmdStream );
	printf ( "Set copyright in %s to \"%s\"\n", kFormNames[formMask], value.c_str() );
	
	if ( formMask & kFormExif ) {
		if ( tiffManager == 0 ) tiffManager = new TIFF_FileWriter;
		tiffManager->SetTag_ASCII ( kTIFF_PrimaryIFD, kTIFF_Copyright, value.c_str() );
	}
	
	if ( formMask & kFormIIM ) {
		if ( iimManager == 0 ) iimManager = new IPTC_Writer;
		iimManager->SetDataSet_UTF8 ( kIPTC_CopyrightNotice, value.c_str(), value.size() );
	}
	
	if ( formMask & kFormXMP ) {
		if ( xmpManager == 0 ) xmpManager = new SXMPMeta;
		xmpManager->SetLocalizedText ( kXMP_NS_DC, "rights", "", "x-default", value.c_str() );
	}

}	// SetCopyright

// =================================================================================================

static void SetCreator ( std::string & cmdStream, XMP_Uns8 formMask )
{
	std::vector<std::string> values = GetStringList ( cmdStream );
	printf ( "Set creator in %s to (", kFormNames[formMask] );
	for ( size_t i = 0; i < values.size(); ++i ) printf ( " \"%s\"", values[i].c_str() );
	printf ( " )\n" );
	
	if ( formMask & kFormExif ) {
		if ( tiffManager == 0 ) tiffManager = new TIFF_FileWriter;
		tiffManager->DeleteTag ( kTIFF_PrimaryIFD, kTIFF_Artist );
		for ( size_t i = 1; i < values.size(); ++i ) {
			values[0] += "; ";	// Multiple Exif Artist values are a single string with "; " separators.
			values[0] += values[i];
		}
		tiffManager->SetTag_ASCII ( kTIFF_PrimaryIFD, kTIFF_Artist, values[0].c_str() );
	}
	
	if ( formMask & kFormIIM ) {
		if ( iimManager == 0 ) iimManager = new IPTC_Writer;
		iimManager->DeleteDataSet ( kIPTC_Creator );
		for ( size_t i = 0; i < values.size(); ++i ) {
			iimManager->SetDataSet_UTF8 ( kIPTC_Creator, values[i].c_str(), values[i].size(), i );
		}
	}
	
	if ( formMask & kFormXMP ) {
		if ( xmpManager == 0 ) xmpManager = new SXMPMeta;
		xmpManager->DeleteProperty ( kXMP_NS_DC, "creator" );
		for ( size_t i = 0; i < values.size(); ++i ) {
			xmpManager->AppendArrayItem ( kXMP_NS_DC, "creator", kXMP_PropArrayIsOrdered, values[i].c_str() );
		}
	}

}	// SetCreator

// =================================================================================================

static void SetKeywords ( std::string & cmdStream, XMP_Uns8 formMask )
{
	std::vector<std::string> values = GetStringList ( cmdStream );
	printf ( "Set keywords in %s to (", kFormNames[formMask] );
	for ( size_t i = 0; i < values.size(); ++i ) printf ( " \"%s\"", values[i].c_str() );
	printf ( " )\n" );
	if ( ! (formMask & (kFormXMP | kFormIIM)) ) { printf ( "  ** Keywords are only in XMP and IIM.\n" ); return; }
	
	if ( formMask & kFormIIM ) {
		if ( iimManager == 0 ) iimManager = new IPTC_Writer;
		iimManager->DeleteDataSet ( kIPTC_Keyword );
		for ( size_t i = 0; i < values.size(); ++i ) {
			iimManager->SetDataSet_UTF8 ( kIPTC_Keyword, values[i].c_str(), values[i].size(), i );
		}
	}
	
	if ( formMask & kFormXMP ) {
		if ( xmpManager == 0 ) xmpManager = new SXMPMeta;
		xmpManager->DeleteProperty ( kXMP_NS_DC, "subject" );
		for ( size_t i = 0; i < values.size(); ++i ) {
			xmpManager->AppendArrayItem ( kXMP_NS_DC, "subject", kXMP_PropArrayIsUnordered, values[i].c_str() );
		}
	}

}	// SetKeywords

// =================================================================================================

enum { kCreateDate = 1, kDigitizeDate = 2, kModifyDate = 4 };

static void SetDates ( std::string & cmdStream, XMP_Uns8 formMask, XMP_Uns8 itemMask )
{
	std::string cmdValue = GetWord ( cmdStream );
	printf ( "Set date(s) in %s to %s\n", kFormNames[formMask], cmdValue.c_str() );
	
	if ( cmdValue == "RIGHT" ) {
	
		// Set the date/time items to specific "right" values.
	
		if ( formMask & kFormExif ) {

			if ( tiffManager == 0 ) tiffManager = new TIFF_FileWriter;

			if ( itemMask & kCreateDate ) {
				tiffManager->SetTag_ASCII ( kTIFF_ExifIFD, kTIFF_DateTimeOriginal, "2001:01:01 01:11:11" );
				tiffManager->SetTag_ASCII ( kTIFF_ExifIFD, kTIFF_SubSecTimeOriginal, "111" );
			}
			if ( itemMask & kDigitizeDate ) {
				tiffManager->SetTag_ASCII ( kTIFF_ExifIFD, kTIFF_DateTimeDigitized, "2001:02:02 01:22:22" );
				tiffManager->SetTag_ASCII ( kTIFF_ExifIFD, kTIFF_SubSecTimeDigitized, "122" );
			}
			if ( itemMask & kModifyDate ) {
				tiffManager->SetTag_ASCII ( kTIFF_PrimaryIFD, kTIFF_DateTime, "2001:03:03 01:33:33" );
				tiffManager->SetTag_ASCII ( kTIFF_ExifIFD, kTIFF_SubSecTime, "133" );
			}

		}
		
		if ( formMask & kFormIIM ) {

			if ( iimManager == 0 ) iimManager = new IPTC_Writer;

			if ( itemMask & kCreateDate ) {
				iimManager->SetDataSet_UTF8 ( kIPTC_DateCreated, "20020101", 8 );
				iimManager->SetDataSet_UTF8 ( kIPTC_TimeCreated, "021111+0100", 11 );
			}
			if ( itemMask & kDigitizeDate ) {
				iimManager->SetDataSet_UTF8 ( kIPTC_DigitalCreateDate, "20020202", 8 );
				iimManager->SetDataSet_UTF8 ( kIPTC_DigitalCreateTime, "022222+0200",11 );
			}

		}
		
		if ( formMask & kFormXMP ) {

			if ( xmpManager == 0 ) xmpManager = new SXMPMeta;

			if ( itemMask & kCreateDate ) {
				xmpManager->SetProperty ( kXMP_NS_Photoshop, "DateCreated", "2003-01-01T03:11:11.311+01:00" );
			}
			if ( itemMask & kDigitizeDate ) {
				xmpManager->SetProperty ( kXMP_NS_XMP, "CreateDate", "2003-02-02T03:22:22.322+02:00" );
			}
			if ( itemMask & kModifyDate ) {
				xmpManager->SetProperty ( kXMP_NS_XMP, "ModifyDate", "2003-03-03T03:33:33.333+03:00" );
			}

		}

	} else if ( cmdValue == "WRONG" ) {
	
		// Set the date/time items to specific "wrong" values.
	
		if ( formMask & kFormExif ) {

			if ( tiffManager == 0 ) tiffManager = new TIFF_FileWriter;

			if ( itemMask & kCreateDate ) {
				tiffManager->SetTag_ASCII ( kTIFF_ExifIFD, kTIFF_DateTimeOriginal, "1991:01:01 01:11:11" );
				tiffManager->SetTag_ASCII ( kTIFF_ExifIFD, kTIFF_SubSecTimeOriginal, "111" );
			}
			if ( itemMask & kDigitizeDate ) {
				tiffManager->SetTag_ASCII ( kTIFF_ExifIFD, kTIFF_DateTimeDigitized, "1991:02:02 01:22:22" );
				tiffManager->SetTag_ASCII ( kTIFF_ExifIFD, kTIFF_SubSecTimeDigitized, "122" );
			}
			if ( itemMask & kModifyDate ) {
				tiffManager->SetTag_ASCII ( kTIFF_PrimaryIFD, kTIFF_DateTime, "1991:03:03 01:33:33" );
				tiffManager->SetTag_ASCII ( kTIFF_ExifIFD, kTIFF_SubSecTime, "133" );
			}

		}
		
		if ( formMask & kFormIIM ) {

			if ( iimManager == 0 ) iimManager = new IPTC_Writer;

			if ( itemMask & kCreateDate ) {
				iimManager->SetDataSet_UTF8 ( kIPTC_DateCreated, "19920101", 8 );
				iimManager->SetDataSet_UTF8 ( kIPTC_TimeCreated, "021111+0100", 11 );
			}
			if ( itemMask & kDigitizeDate ) {
				iimManager->SetDataSet_UTF8 ( kIPTC_DigitalCreateDate, "19920202", 8 );
				iimManager->SetDataSet_UTF8 ( kIPTC_DigitalCreateTime, "022222+0200",11 );
			}
			if ( itemMask & kModifyDate ) {
				// There is no IIM modify date.
			}

		}
		
		if ( formMask & kFormXMP ) {

			if ( xmpManager == 0 ) xmpManager = new SXMPMeta;

			if ( itemMask & kCreateDate ) {
				xmpManager->SetProperty ( kXMP_NS_Photoshop, "DateCreated", "1993-01-01T03:11:11.311+01:00" );
			}
			if ( itemMask & kDigitizeDate ) {
				xmpManager->SetProperty ( kXMP_NS_XMP, "CreateDate", "1993-02-02T03:22:22.322+02:00" );
			}
			if ( itemMask & kModifyDate ) {
				xmpManager->SetProperty ( kXMP_NS_XMP, "ModifyDate", "1993-03-03T03:33:33.333+03:00" );
			}

		}

	} else {
	
		// Set the date/time items to the single given value. The timezone is optional, but
		// SXMPUtils::ConvertToDate does not yet report if it is missing. The possible forms are:
		//   YYYY
		//   YYYY-MM
		//   YYYY-MM-DD
		//   YYYY-MM-DDThh:mm
		//   YYYY-MM-DDThh:mm:ssTZD
		//   YYYY-MM-DDThh:mm:ss.sTZD

		XMP_DateTime binValue;
		
		try {
			SXMPUtils::ConvertToDate ( cmdValue.c_str(), &binValue );
		} catch ( ... ) {
			printf ( "  ** Date must be in ISO 8601 form.\n" );
			return;
		}
	
		bool hasTZ = false;
		if ( (cmdValue.size() > 20) && (cmdValue[19] == '.') ) {	// Form with fractional seconds.
			for ( size_t i = 20; i < cmdValue.size(); ++i ) {
				if ( (cmdValue[i] == '+') || (cmdValue[i] == '-') ) { hasTZ = true; break; }
			}
		} else if ( (cmdValue.size() > 19) || ((cmdValue.size() > 16) && (cmdValue[16] != ':')) ) {
			hasTZ = true;
		}

		char buffer[32];	// Big enough for both Exif and IIM forms.
		
		if ( formMask & kFormExif ) {
	
			if ( tiffManager == 0 ) tiffManager = new TIFF_FileWriter;
	
			if ( cmdValue.size() == 4 ) {			// Just "YYYY:  :     :  :  "
				sprintf ( buffer, "%04d:  :     :  :  ", binValue.year );
			} else if ( cmdValue.size() == 7 ) {	// Just "YYYY:MM:     :  :  "
				sprintf ( buffer, "%04d:%02d:     :  :  ", binValue.year, binValue.month );
			} else if ( cmdValue.size() == 10 ) {	// Just "YYYY:MM:DD   :  :  "
				sprintf ( buffer, "%04d:%02d:%02d   :  :  ", binValue.year, binValue.month, binValue.day );
			} else if ( cmdValue.size() == 16 ) {	// Just "YYYY:MM:DD HH:MM:  "
				sprintf ( buffer, "%04d:%02d:%02d %02d:%02d:  ",
						  binValue.year, binValue.month, binValue.day, binValue.hour, binValue.minute );
			} else {								// Full "YYYY:MM:DD HH:MM:SS".
				sprintf ( buffer, "%04d:%02d:%02d %02d:%02d:%02d",
						  binValue.year, binValue.month, binValue.day,
						  binValue.hour, binValue.minute, binValue.second );
			}

			if ( itemMask & kCreateDate ) {
				tiffManager->SetTag_ASCII ( kTIFF_ExifIFD, kTIFF_DateTimeOriginal, buffer );
			}
			if ( itemMask & kDigitizeDate ) {
				tiffManager->SetTag_ASCII ( kTIFF_ExifIFD, kTIFF_DateTimeDigitized, buffer );
			}
			if ( itemMask & kModifyDate ) {
				tiffManager->SetTag_ASCII ( kTIFF_PrimaryIFD, kTIFF_DateTime, buffer );
			}
	
			if ( binValue.nanoSecond == 0 ) {

				if ( itemMask & kCreateDate ) {
					tiffManager->DeleteTag ( kTIFF_ExifIFD, kTIFF_SubSecTimeOriginal );
				}
				if ( itemMask & kDigitizeDate ) {
					tiffManager->DeleteTag ( kTIFF_ExifIFD, kTIFF_SubSecTimeDigitized );
				}
				if ( itemMask & kModifyDate ) {
					tiffManager->DeleteTag ( kTIFF_ExifIFD, kTIFF_SubSecTime );
				}

			} else {

				sprintf ( buffer, "%d", binValue.nanoSecond );
				for ( size_t i = (strlen(buffer) - 1); ((i > 0) && (buffer[i] == '0')); --i ) buffer[i] = '\0';
				if ( itemMask & kCreateDate ) {
					tiffManager->SetTag_ASCII ( kTIFF_ExifIFD, kTIFF_SubSecTimeOriginal, buffer );
				}
				if ( itemMask & kDigitizeDate ) {
					tiffManager->SetTag_ASCII ( kTIFF_ExifIFD, kTIFF_SubSecTimeDigitized, buffer );
				}
				if ( itemMask & kModifyDate ) {
					tiffManager->SetTag_ASCII ( kTIFF_ExifIFD, kTIFF_SubSecTime, buffer );
				}

			}
	
		}
		
		if ( formMask & kFormIIM ) {

			if ( iimManager == 0 ) iimManager = new IPTC_Writer;

			sprintf ( buffer, "%04d%02d%02d", binValue.year, binValue.month, binValue.day );

			if ( itemMask & kCreateDate ) {
				iimManager->SetDataSet_UTF8 ( kIPTC_DateCreated, buffer, strlen(buffer) );
			}
			if ( itemMask & kDigitizeDate ) {
				iimManager->SetDataSet_UTF8 ( kIPTC_DigitalCreateDate, buffer, strlen(buffer) );
			}

			if ( cmdValue.size() <= 10 ) {

				if ( itemMask & kCreateDate ) {
					iimManager->DeleteDataSet ( kIPTC_TimeCreated );
				}
				if ( itemMask & kDigitizeDate ) {
					iimManager->DeleteDataSet ( kIPTC_DigitalCreateTime );
				}
			
			} else {
			
				if ( hasTZ ) {
					sprintf ( buffer, "%02d%02d%02d%c%02d%02d",
							  binValue.hour, binValue.minute, binValue.second,
							  ((binValue.tzSign < 0) ? '-' : '+'), binValue.tzHour, binValue.tzMinute );
				} else {
					sprintf ( buffer, "%02d%02d%02d", binValue.hour, binValue.minute, binValue.second );
				}
	
				if ( itemMask & kCreateDate ) {
					iimManager->SetDataSet_UTF8 ( kIPTC_TimeCreated, buffer, strlen(buffer) );
				}
				if ( itemMask & kDigitizeDate ) {
					iimManager->SetDataSet_UTF8 ( kIPTC_DigitalCreateTime, buffer, strlen(buffer) );
				}
				
			}

		}
		
		if ( formMask & kFormXMP ) {

			if ( xmpManager == 0 ) xmpManager = new SXMPMeta;

			if ( itemMask & kCreateDate ) {
				xmpManager->SetProperty ( kXMP_NS_Photoshop, "DateCreated", cmdValue.c_str() );
			}
			if ( itemMask & kDigitizeDate ) {
				xmpManager->SetProperty ( kXMP_NS_XMP, "CreateDate", cmdValue.c_str() );
			}
			if ( itemMask & kModifyDate ) {
				xmpManager->SetProperty ( kXMP_NS_XMP, "ModifyDate", cmdValue.c_str() );
			}

		}

	}
	
}	// SetDates

// =================================================================================================

static void SetRating ( std::string & cmdStream, XMP_Uns8 formMask )
{
	std::string value = GetWord ( cmdStream );
	printf ( "Set rating in %s to %s\n", kFormNames[formMask], value.c_str() );
	if ( ! (formMask & kFormXMP) ) { printf ( "  ** Rating is only in XMP.\n" ); return; }
	
	if ( xmpManager == 0 ) xmpManager = new SXMPMeta;
	xmpManager->SetProperty ( kXMP_NS_XMP, "Rating", value.c_str() );

}	// SetRating

// =================================================================================================

static void SetOrientation ( std::string & cmdStream, XMP_Uns8 formMask )
{
	XMP_Uns32 value = GetUInt ( cmdStream );
	printf ( "Set orientation in %s to %d\n", kFormNames[formMask], value );
	if ( ! (formMask & kFormExif) ) { printf ( "  ** Orientation is only in Exif.\n" ); return; }

	if ( tiffManager == 0 ) tiffManager = new TIFF_FileWriter;
	tiffManager->SetTag_Short ( kTIFF_PrimaryIFD, kTIFF_Orientation, value );
	if ( tiffManager->GetTag ( kTIFF_TNailIFD, kTIFF_Orientation, 0 ) ) {
		tiffManager->SetTag_Short ( kTIFF_TNailIFD, kTIFF_Orientation, value );
	}

}	// SetOrientation

// =================================================================================================

static void SetUserComment ( std::string & cmdStream )
{
	std::string ucForm = GetWord ( cmdStream );
	std::string asciiValue = GetString ( cmdStream );
	printf ( "Set UserComment as %s to \"%s\".\n", ucForm.c_str(), asciiValue.c_str() );

	if ( tiffManager == 0 ) tiffManager = new TIFF_FileWriter;
	
	if ( ucForm == "ASCII" ) {
		asciiValue.insert ( 0, "ASCII\x0\x0\x0", 8 );
		tiffManager->SetTag ( kTIFF_ExifIFD, kTIFF_UserComment, kTIFF_UndefinedType, asciiValue.size(), asciiValue.c_str() );
		return;
	}
	
	bool bigEndian, useBOM;
	if ( ucForm == "BE-BOM" ) {
		bigEndian = true;
		useBOM = false;
	} else if ( ucForm == "BE+BOM" ) {
		bigEndian = true;
		useBOM = true;
	} else if ( ucForm == "LE-BOM" ) {
		bigEndian = false;
		useBOM = false;
	} else if ( ucForm == "LE+BOM" ) {
		bigEndian = false;
		useBOM = true;
	} else {
		printf ( "  ** Need to tell endianness and BOMness\n" );
	}
	
	std::string u16Value = "UNICODE ";
	u16Value[7] = '\0';
	
	if ( bigEndian ) {
		if ( useBOM ) u16Value += "\xFE\xFF";
		for ( size_t i = 0; i < asciiValue.size(); ++i ) { u16Value += '\0'; u16Value += asciiValue[i]; }
	} else {
		if ( useBOM ) u16Value += "\xFF\xFE";
		for ( size_t i = 0; i < asciiValue.size(); ++i ) { u16Value += asciiValue[i]; u16Value += '\0'; }
	}

	tiffManager->SetTag ( kTIFF_ExifIFD, kTIFF_UserComment, kTIFF_UndefinedType, u16Value.size(), u16Value.c_str() );

}	// SetUserComment

// =================================================================================================

static void SetItem ( std::string & cmdStream )
{
	std::string dataForm = GetWord ( cmdStream );
	std::string itemName = GetWord ( cmdStream );
	
	XMP_Uns8 formMask = 0;

	if ( dataForm == "usercomment" ) {
		SetUserComment ( cmdStream );
		return;
	}

	if ( dataForm == "exif" ) {
		formMask = kFormExif;
	} else if ( dataForm == "iim" ) {
		formMask = kFormIIM;
	} else if ( dataForm == "xmp" ) {
		formMask = kFormXMP;
	} else if ( dataForm == "all" ) {
		formMask = kFormExif | kFormIIM | kFormXMP;
	} else {
		printf ( "** Unrecognized data form: \"%s\".\n", dataForm.c_str() );
		return;
	}
	
	if ( itemName == "description" ) {
		SetDescription ( cmdStream, formMask );
	} else if ( itemName == "copyright" ) {
		SetCopyright ( cmdStream, formMask );
	} else if ( itemName == "creator" ) {
		SetCreator ( cmdStream, formMask );
	} else if ( itemName == "keywords" ) {
		SetKeywords ( cmdStream, formMask );
	} else if ( itemName == "c-date" ) {
		SetDates ( cmdStream, formMask, kCreateDate );
	} else if ( itemName == "d-date" ) {
		SetDates ( cmdStream, formMask, kDigitizeDate );
	} else if ( itemName == "m-date" ) {
		SetDates ( cmdStream, formMask, kModifyDate );
	} else if ( itemName == "dates" ) {
		SetDates ( cmdStream, formMask, (kCreateDate | kDigitizeDate | kModifyDate) );
	} else if ( itemName == "rating" ) {
		SetRating ( cmdStream, formMask );
	} else if ( itemName == "orientation" ) {
		SetOrientation ( cmdStream, formMask );
	} else {
		printf ( "** Unrecognized MWG item: \"%s\"\n", itemName.c_str() );
	}
	
}	// SetItem

// =================================================================================================
// =================================================================================================
// Top level routines
// =================================================================================================
// =================================================================================================

static void ProcessCommands ( std::string & cmdStream, const char * whichFile = "" )
{
	std::string nextCmd;
	
	while ( ! cmdStream.empty() ) {
	
		nextCmd = GetWord ( cmdStream );
		if ( nextCmd.empty() ) break;
		
		if ( nextCmd == "read" ) {

			std::string filePath = GetString ( cmdStream );

			bool haveWildCard = ReadFile ( filePath, "jpg" );	// Do JPEG first for wildcard.

			if ( haveWildCard && (whichFile[0] == 0) ) {
				std::string tempStream = cmdStream;
				ProcessCommands ( tempStream, "jpg" );	// Process a copy of the commands for JPEG.
				whichFile = "tif";	// Process the commands again for TIFF.
				ReadFile ( filePath, "tif" );
			}

		} else if ( nextCmd == "write" ) {

			WriteFile ( cmdStream, kNormalIIMDigest, whichFile );
			if ( strcmp ( whichFile, "jpg" ) == 0 ) return;	// Exit the nested command portion.
			if ( strcmp ( whichFile, "tif" ) == 0 ) whichFile = "";	// Prepare for the next wildcard read.

		} else if ( nextCmd == "write-nd" ) {

			WriteFile ( cmdStream, kNoIIMDigest, whichFile );
			if ( strcmp ( whichFile, "jpg" ) == 0 ) return;	// Exit the nested command portion.
			if ( strcmp ( whichFile, "tif" ) == 0 ) whichFile = "";	// Prepare for the next wildcard read.

		} else if ( nextCmd == "write-zd" ) {

			WriteFile ( cmdStream, kZeroIIMDigest, whichFile );
			if ( strcmp ( whichFile, "jpg" ) == 0 ) return;	// Exit the nested command portion.
			if ( strcmp ( whichFile, "tif" ) == 0 ) whichFile = "";	// Prepare for the next wildcard read.

		} else if ( nextCmd == "write-fd" ) {

			WriteFile ( cmdStream, kFakeIIMDigest, whichFile );
			if ( strcmp ( whichFile, "jpg" ) == 0 ) return;	// Exit the nested command portion.
			if ( strcmp ( whichFile, "tif" ) == 0 ) whichFile = "";	// Prepare for the next wildcard read.

		} else if ( nextCmd == "strip" ) {

			StripBlock ( cmdStream );

		} else if ( nextCmd == "delete" ) {

			DeleteItem ( cmdStream );

		} else if ( nextCmd == "set" ) {

			SetItem ( cmdStream );

		} else if ( nextCmd == "--" ) {

			while ( ! cmdStream.empty() ) {
				if ( (cmdStream[0] == '\n') || (cmdStream[0] == '\r') ) break;
				cmdStream.erase ( 0, 1 );
			}

		} else {

			printf ( "** Unrecognized command: \"%s\"\n", nextCmd.c_str() );
			return;

		}
	
	}

}	// ProcessCommands

// =================================================================================================

extern "C" int main ( int argc, const char * argv [] )
{
	std::string cmdStream;

	if ( argc > 1 ) {
		for ( int i = 1; i < argc; ++i ) { cmdStream += argv[i]; cmdStream += ' '; }
	} else {
		char buffer[4096];
		while ( true ) {
			size_t count = fread ( buffer, 1, sizeof(buffer), stdin );
			if ( count == 0 ) break;
			cmdStream.append ( buffer, count );
		}
	}
	
	(void) SXMPMeta::Initialize();
	ProcessCommands ( cmdStream );
	SXMPMeta::Terminate();
	
	return 0;

}	// main
